<?php

namespace App\Http\Resources\Base;

use Illuminate\Http\Resources\Json\JsonResource;

class TranslatableResource extends JsonResource
{
    protected string $lang;

    public function __construct($resource)
    {
        parent::__construct($resource);
        $this->lang = request()->header('lang', 'ar'); // default: ar
    }

    /**
     * ترجمة حقل JSON تلقائيًا
     * يدعم: {"ar": "...", "en": "..."}
     */
    protected function translate($field, $fallback = null)
    {
        $value = $this->{$field};

        // لو الحقل مش موجود أصلاً
        if (is_null($value)) {
            return $fallback;
        }

        // لو هو string JSON
        if (is_string($value)) {
            $decoded = json_decode($value, true);

            if (is_array($decoded) && isset($decoded[$this->lang])) {
                return $decoded[$this->lang];
            }

            // لو JSON مش صحيح أو مفيش المفتاح، ارجع القيمة الأصلية
            return $value;
        }

        // لو كان array أصلًا (نادرًا لكن ممكن)
        if (is_array($value) && isset($value[$this->lang])) {
            return $value[$this->lang];
        }

        return $value ?? $fallback;
    }

    /**
     * ترجمة عدة حقول مرة واحدة
     * مثال: $this->translateFields(['title', 'description', 'name'])
     */
    protected function translateFields(array $fields, array $customKeys = [])
    {
        $data = [];

        foreach ($fields as $field) {
            $key = $customKeys[$field] ?? $field;
            $data[$key] = $this->translate($field);
        }

        return $data;
    }

    /**
     * Wrap الـ response تلقائيًا في data + success
     */
    public function toArray($request)
    {
        return [
            'data' => parent::toArray($request),
        ];
    }

    public function with($request)
    {
        return [
            'success' => true,
            'message' => 'تم جلب البيانات بنجاح',
        ];
    }

    /**
     * Helper: ارجع رابط كامل للصور
     */
    protected function imageUrl($path, $default = null)
    {
        if (!$path) return $default;
        return asset('storage/' . ltrim($path, '/'));
    }
}
